package FEJDMath;

public class MathStuff
{
	/** Thread used to assemble the matrix */
	private static MakeMatThread matThread;
	/** Thread used to simulate the boundary conditions of type 1 */
	private static SimulBCT1Thread bct1Thread;
	/** Thread used to compute the boundary conditions of type 2 */
	private static CalcBCT2Thread bct2Thread;
	/** Thread used to compute the boundary conditions of type 3 */
	private static CalcBCT3Thread bct3Thread;
	/** Thread used to compute the solution */
	private static SolutionThread solThread;
	/** The mesh we are working with */
	private static Mesh m;
	/** The stiffnes matrix stored in a skyline format */
	private static SkyLine stiffnessMatrix;
	/** The last vector */
	private static double[] lastVector;

	/** Creates and assembles the stiffness matrix */
	public static void makeMat() {
		int size = m.getSize();
		lastVector = new double[size];
		stiffnessMatrix = new SkyLine(size);

		// ths stiffnessMatrix is already set to 0
		for(int i=0; i < size; i++)
			lastVector[i]=0;

		matThread = new MakeMatThread(m);
		matThread.start();
	}
	
	/** Returns true if the thread which is making the matrix is alive */
	public static boolean isMatThreadAlive() {
		return matThread.isAlive();
	}

	/** Returns true if the thread which is simulating the computation of the boundary conditions of type 1 is alive */
	public static boolean isBCT1ThreadAlive() {
		return bct1Thread.isAlive();
	}

	/** Returns true if the thread which is computing the boundary conditions of type 2 is alive */
	public static boolean isBCT2ThreadAlive() {
		return bct2Thread.isAlive();
	}

	/** Returns true if the thread which is computing the boundary conditions of type 3 is alive */
	public static boolean isBCT3ThreadAlive() {
		return bct3Thread.isAlive();
	}

	/** Returns true if the thread which is computing the solution is alive */
	public static boolean isSolThreadAlive() {
		return solThread.isAlive();
	}

	/** Computes the boundary condition of 2nd type */
	public static void calcBCT2() {
		bct2Thread = new CalcBCT2Thread(m);
		bct2Thread.start();
	}

	/** Computes the boundary condition of 3rd type */
	public static void calcBCT3() {
		bct3Thread = new CalcBCT3Thread(m);
		bct3Thread.start();
	}

	/** Simulates boundary conditions 1st type, as if they where computed at the end
	and sets the lastvector to the right value */
	public static void simulBCT1() {
		bct1Thread = new SimulBCT1Thread(m, lastVector, stiffnessMatrix);
		bct1Thread.start();
	}

	/** Displays the matric ant the last vector, useful for debugging */
	public static void printDebug()
	{
		// ------------ begin debugging info
		System.out.println("Debugging info");
		//System.out.println(stiffnessMatrix); // may take VERY long
		System.out.println(stiffnessMatrix.toString2());

		System.out.println("LastVector");
		for(int i =0; i < m.getSize(); i++)
		{
			System.out.print(lastVector[i] + " ");
			if(i%10 == 0 && i != 0)
				System.out.print("\n");
		}
		// ------------ end debugging info
	}

	/** Solves the problem and sets the node to their final temperature */
	public static void solvePb() {
		solThread = new SolutionThread(m, lastVector, stiffnessMatrix);
		solThread.start();
	}


	/** Adds x to the element (i,j) of the stiffness matrix */
	public static void addSMatrix(int i, int j, double x) {
		if(x != 0) {
			double t = stiffnessMatrix.get(i,j);
			stiffnessMatrix.set(i,j,t+x);
		}
	}

	/** Retuns the (i,j) element of the stiffness matrix */
	public static double getMatrixValue(int i, int j)
	{
		return stiffnessMatrix.get(i,j);
	}

	/** Retuns the (i,j) element of the last vector */
	public static double getVectorValue(int i)
	{
		return lastVector[i];
	}

	/** Adds x to the ith element of the last vector */
	public static void addLVector(int i, double x) {
		lastVector[i] += x;
	}

	/** Sets the mesh */
	public static void setMesh(Mesh nm) {
		m = nm;
	}

	/** Returns the mesh */
	public static Mesh getMesh() {
		return m;
	}

	/** Sets all the temperatures of the mesh to 0 */
	public static void clearAllTemp() {
		for(int i=0; i < m.getNbOfMeshPoints(); i++)
			m.getMeshPoint(i).setTemperature(0);
	}

	/** Returns the size of the shape of the matrix */
	public static int getSkylineSize() {
		return stiffnessMatrix.getSize();
	}

}
